<?php

namespace App\Twig\Extension;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use Twig\TwigFilter;
use Pimcore\File;
use App\Tool\Utils;

class UtilsExtension extends AbstractExtension
{
	/**
	 * @return array|TwigFunction[]
	 */
	public function getFunctions()
	{
		return [
			new TwigFunction('utils_url', [$this, 'url']),
			new TwigFunction('utils_image', [$this, 'utilsImage'], ['is_safe' => ['html']]),
			new TwigFunction('utils_thumbnail', [$this, 'utilsThumbnail'], ['is_safe' => ['html']]),
			new TwigFunction('utils_cut_string', [$this, 'utilsCutString']),
			new TwigFunction('utils_encode', [$this, 'utilsEncode']),
			new TwigFunction('utils_webalize', [$this, 'utilsWebalize']),
			new TwigFunction('utils_get_file_extension', [$this, 'utilsGetFileExtension']),
			new TwigFunction('utils_format_size_units', [$this, 'utilsFormatSizeUnits']),
			new TwigFunction('utils_call_static', [$this, 'utilsCallStatic']),
			new TwigFunction('utils_get_from_runtime_cache', [$this, 'getFromRuntimeCache']),
		];
	}

	public function getFilters()
	{
		return [
			new TwigFilter('safe_email', [$this, 'safeEmail']),
			new TwigFilter('typo', [$this, 'typo'], ['is_safe' => ['html']]),
			new TwigFilter('ucfirst', [$this, 'ucfirst'], ['is_safe' => ['html']]),
		];
	}

	/**
	 * Returns an email converted to ASCII characters
	 *
	 * @param string
	 * @return string
	 */
	public function safeEmail($email)
	{
		return str_replace(['@', '.'], ['&#x00040;', '&#x0002E;'], $email);
	}

	/**
	 * Typography
	 *
	 * @param string
	 * @return string
	 */
	public function typo(?string $text): string
	{
		$filtered = preg_replace(['/(\s+|&nbsp;)(a|o|u|i|k|s|v|z)(\s+)/i'], ['$1$2&nbsp;'], $text);
		$filtered = preg_replace('/<p[^>]*>([\s]|&nbsp;)*<\/p>/', '', $filtered);
		return $filtered;
	}

	/**
	 * @param string
	 * @return string
	 */
	public function ucfirst(?string $text): string
	{
		return ucfirst($text);
	}

	/**
	 * Returns a document URL by document path, based on current request language.
	 *
	 * @param \Twig\Environment $context
	 *
	 * @return string
	 */
	public function url($staticRoute, $params)
	{
		return Utils::url($staticRoute, $params);
	}

	/**
	 * Returns a HTML $image element (as a $thumbnail).
	 *
	 * @param Document\Editable\Image|DataObject\Data\Hotspotimage|Asset\Image $image
	 * @param string $thumbnail
	 * @param Asset\Image|string|int $fallbackImage
	 * @param array $imgOptions
	 * @param array $pictureOptions
	 *
	 * @return string
	 */
	public function utilsImage($image, $thumbnail, $fallbackImage = null, $imgOptions = [], $pictureOptions = [])
	{
		return Utils::image($image, $thumbnail, $fallbackImage, $imgOptions, $pictureOptions);
	}

	/**
	 * Returns a path to $image thumbnail
	 *
	 * @param Document\Editable\Image|DataObject\Data\Hotspotimage|Asset\Image $image
	 * @param string $thumbnail
	 * @param Asset\Image|string|int $fallbackImage
	 *
	 * @return string
	 */
	public function utilsThumbnail($image, $thumbnail, $fallbackImage = null)
	{
		return Utils::thumbnail($image, $thumbnail, $fallbackImage);
	}

	/**
	 * Cuts string to $length, respecting whitespace.
	 *
	 * @param string $string
	 * @param int $length
	 * @param string $suffix
	 *
	 * @return string
	 */
	public function utilsCutString($string, $length, $suffix = '...')
	{
		return Utils::cutStringRespectingWhitespace($string, $length, $suffix);
	}

	/**
	 * Encodes a string depending on context.
	 *
	 * @param string $string String to encode
	 * @param boolean $isForJavaScript Output is in JavaScript
	 * @param boolean $isForJavaScriptHTMLAttribute Output is in JavaScript, in a HTML attribute
	 * @return string
	 */
	public function utilsEncode(string $string, $isForJavaScript = false, $isForJavaScriptHTMLAttribute = false)
	{
		if ($isForJavaScript) {
			if ($isForJavaScriptHTMLAttribute) {
				return htmlspecialchars(json_encode($string), ENT_QUOTES);
			} else {
				return json_encode($string);
			}
		}
		return htmlspecialchars($string, ENT_QUOTES);
	}

	/**
	 * Returns webalized / slugified string.
	 *
	 * @param mixed $path
	 * @return string
	 */
	public function utilsWebalize($path)
	{
		if (!$path) {
			return '';
		}
		return Utils::webalize($path);
	}

	/**
	 * Returns file extension for $filename.
	 *
	 * @param string $filename
	 * @return string
	 */
	public function utilsGetFileExtension(string $filename)
	{
		return File::getFileExtension($filename);
	}

	/**
	 * Returns $size formatted with units as a string.
	 *
	 * @param int $size
	 * @return string
	 */
	public function utilsFormatSizeUnits(int $size)
	{
		return Utils::formatSizeUnits($size);
	}

	/**
	 * Calls static class method
	 *
	 */
	public function utilsCallStatic(string $class, string $method, ...$args)
	{
		if (!class_exists($class)) {
			throw new \Exception("Cannot call static method $method on Class $class: Invalid Class");
		}

		if (!method_exists($class, $method)) {
			throw new \Exception("Cannot call static method $method on Class $class: Invalid method");
		}

		return forward_static_call_array([$class, $method], $args);

	}

	/**
	 * Returns $size formatted with units as a string.
	 *
	 * @param string $key
	 * @return mixed
	 */
	public function getFromRuntimeCache(string $key)
	{
		return \Pimcore\Cache\Runtime::isRegistered($key)
			? \Pimcore\Cache\Runtime::get($key)
			: null;
	}
}
