<?php

namespace AppBundle\Service;

use Pimcore\Logger;
use Pimcore\Model\Asset;
use Pimcore\Model\Document;

class MailManager
{
	/**
	 * @var Document\Service
	 */
	protected $documentService;

	/**
	 * @param Document\Service
	 */
	public function __construct(Document\Service $documentService)
	{
		$this->documentService = $documentService;
	}

	/**
	 * @param Document\Email|string  $pathOrTemplate
	 * @param string                 $language
	 * @param string                 $to
	 * @param string                 $from
	 * @param array                  $params
	 * @param \Swift_Mime_Attachment $attachment
	 * @param string                 $replyTo
	 *
	 * @return bool
	 */
	public function sendMail($pathOrTemplate, $language, $to = null, $from = null, $params = null, \Swift_Mime_Attachment $attachment = null, $replyTo = null)
	{
		if ($pathOrTemplate instanceof Document\Email) {
			$document = $pathOrTemplate;
		} else {
			$document = Document\Email::getByPath($pathOrTemplate);
			if (!$document) {
				return false;
			}
			// try to load translated template
			if ($document->getProperty('language') != $language) {
				$translations = $this->documentService->getTranslations($document);
				if (isset($translations[$language])) {
					$translatedDocument = Document\Email::getById($translations[$language]);
					if ($translatedDocument) {
						$document = $translatedDocument;
					}
				}
			}
		}
		if (!$document instanceof Document\Email) {
			return false;
		}

		$mailer = new \Pimcore\Mail();
		$mailer->setCharset('utf-8');

		if ($to) {
			$document->setTo($to);
		}
		if ($from) {
			$document->setFrom($from);
		}

		$mailer->setDocument($document);

		if ($replyTo) {
			$mailer->setReplyTo($replyTo);
		}
		if ($params) {
			$mailer->setParams($params);
		}
		// try to load attachment from "attachment" property if not passed directly
		if (!$attachment) {
			$attachment = $this->loadAttachmentFromProperty($document, $language);
		}
		if ($attachment) {
			$mailer->addAttachment($attachment);
		}

		try {
			$mailer->send();

			return true;
		} catch (\Exception $e) {
			Logger::error('MAIL - '.$e->getMessage()."\n".$e->getTraceAsString());

			return false;
		}
	}

	/**
	 * @param Document\Email $pathOrTemplate
	 * @param string         $language
	 *
	 * @return \Swift_Attachment|null
	 */
	private function loadAttachmentFromProperty($pathOrTemplate, $language)
	{
		if ($pathOrTemplate instanceof Document\Email) {
			$document = $pathOrTemplate;
		} else {
			$document = Document\Email::getByPath($pathOrTemplate);
			if (!$document) {
				return;
			}
			// try to load translated template
			if ($document->getProperty('language') != $language) {
				$translations = $this->documentService->getTranslations($document);
				if (isset($translations[$language])) {
					$translatedDocument = Document\Email::getById($translations[$language]);
					if ($translatedDocument) {
						$document = $translatedDocument;
					}
				}
			}
		}

		$attachment = null;
		$asset = $document->getProperty('attachment');
		if ($asset instanceof Asset) {
			$attachment = \Swift_Attachment::fromPath($asset->getFileSystemPath())
				->setFilename($asset->getFilename());
		}

		return $attachment;
	}
}
