<?php

namespace AppBundle\Tool;

use AppBundle\Model\IElasticObject;
use AppBundle\Model\IElasticObjectFulltext;
use Elasticsearch\ClientBuilder;
use Pimcore\Cache;
use Pimcore\Db;
use Pimcore\Db\Connection;
use Pimcore\Model\Document;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Elasticsearch (re)indexing tooling build on top of the official Elasticsearch php client library
 * 
 * The main idea is to index fulltext content (similar to how google does it) into two separate indices for document pages and object pages
 *  - two indices, because documents and object have separated IDs, which would cause conflicts with only one index
 *  - mapping of these two indices is the same and is defined in the ElasticSearchConfig.php file
 *  - on a multi language setup those incides are futher split per language, because we want to use different analyzers per language to provide better search results
 *  - Page documents are indexed by default, object pages are indexed if the class is extended via Pimcore classmapping and implements the IElasticObjectFulltext interface
 * 
 * Additionally, custom indices can be created, one per class, for other special search use-cases, e.g. filtering of products on e-shops
 *  - these indices are created if the class is extended via Pimcore classmapping and implements the IElasticObject interface
 *  - custom mapping per class can be created in the ElasticSearchConfig.php file (if not, default_mapping is used)
 * 
 * Futhermore, aliases are used to make content reindexing without any search downtime
 *  - aliases point to active indices and are used in queries
 *  - during content reindexing, new indices are created, content is indexed into them, then the aliases are switched and old indices are removed
 * 
 * Complete content reindexing can either be done via elastic:reindex command or by calling the /plugin/backend/reindex-elastic-search endpoint (bound to a button in admin UI)
 * 
 * Partial content reindexing happens everytime a document Page or object implementing IElasticObjectFulltext or IElasticObject is saved
 *  - see the overridden save() methods of Model classes
 */
final class ElasticSearch
{
	const NEW_INDEX_PREFIX_CACHE_KEY = 'elastic_new_index_prefix';
	const DOCUMENT_FULLTEXT_INDEX_KEY = 'document';
	const OBJECT_FULLTEXT_INDEX_KEY = 'object';

	/**
	 * @var \Elasticsearch\Client
	 */
	private static $client = null;

	/**
	 * @var array
	 */
	private static $config = null;

	/**
	 * @var OutputInterface
	 */
	private static $outputInterface = null;

	/**
	 * @var ProgressBar
	 */
	private static $progressBar = null;

	public static function getClient(): \Elasticsearch\Client
	{
		if (!self::$client) {
			self::$client = ClientBuilder::create()
				->setHosts(['http://localhost:9200'])
				->build();
		}

		return self::$client;
	}

	public static function setOutputInterface(OutputInterface $outputInterface = null): void
	{
		self::$outputInterface = $outputInterface;
	}

	private static function write(string $msg): void
	{
		if (self::$outputInterface) {
			self::$outputInterface->write($msg);
		}
	}

	private static function writeln(string $msg): void
	{
		if (self::$outputInterface) {
			self::$outputInterface->writeln($msg);
		}
	}

	private static function progressStart(string $type, int $max): void
	{
		if (self::$outputInterface && $max) {
			self::$progressBar = new ProgressBar(self::$outputInterface, $max);
			self::$progressBar->setFormat(str_pad($type, 20).'  [%bar% %percent:3s%%] %current%/%max% %remaining:6s%');
			self::$progressBar->setRedrawFrequency(ceil($max / 100));
		} else {
			self::$progressBar = null;
		}
	}

	private static function progressAdvance(int $step = 1): void
	{
		if (self::$progressBar) {
			self::$progressBar->advance($step);
		}
	}

	private static function progressEnd(): void
	{
		if (self::$progressBar) {
			self::$progressBar->finish();
			self::writeln(' <info>✔</info>');
		}
	}

	public static function getAliasPrefix(): string
	{
		/** @var Connection $connection */
		$connection = Db::getConnection();
		$alias = $connection->getParams()['dbname'];

		return $alias;
	}

	public static function getIndex(string $suffix, bool $forceCurrent = false): ?string
	{
		$index = null;
		$aliasPrefix = self::getAliasPrefix();
		if ($forceCurrent === false && Cache\Runtime::isRegistered(self::NEW_INDEX_PREFIX_CACHE_KEY)) {
			$indexPrefix = Cache\Runtime::get(self::NEW_INDEX_PREFIX_CACHE_KEY);
			$index = $aliasPrefix . '_' . $indexPrefix . '_' . $suffix;
		} else {
			$alias = $aliasPrefix . '_' . $suffix;
			$elasticClient = self::getClient();
			if ($elasticClient->indices()->existsAlias(['name' => $alias])) {
				$index = key($elasticClient->indices()->getAlias(['name' => $alias]));
			}
		}
		return $index;
	}

	public static function isUpdateRequest(): bool
	{
		return Cache\Runtime::isRegistered('elasticSearchUpdateRequest');
	}

	private static function createFulltextIndices(): array
	{
		$elasticClient = self::getClient();
		$indices = [];

		$fulltextMapping = self::getObjectMapping('fulltext_mapping');
		foreach ($fulltextMapping['meta']['languages'] as $lang) {
			$mapping = [
				'_source' => ['enabled' => true],
				'dynamic' => false,
				'properties' => $fulltextMapping['properties'],
			];
			self::addFulltextFieldsToMapping($mapping, $fulltextMapping['meta']['fulltextFields'], $lang);

			foreach ([self::DOCUMENT_FULLTEXT_INDEX_KEY, self::OBJECT_FULLTEXT_INDEX_KEY] as $key) {
				$suffix = $key.'_'.$lang;
				$indexName = self::getIndex($suffix);
				$elasticClient->indices()->create([
					'index' => $indexName,
					'body' => [
						'settings' => ['analysis' => self::getAnalysis()],
						'mappings' => $mapping,
					]
				]);
				$indices[$suffix] = $indexName;
				self::writeln(sprintf('  <comment>[%s]</comment> <info>✔</info>', $indexName));
			}
		}

		return $indices;
	}

	public static function getObjectMapping(string $className): array
	{
		if (!self::$config) {
			self::$config = include PIMCORE_PROJECT_ROOT.'/src/AppBundle/Tool/ElasticSearchConfig.php';
		}

		$mappingKey = strtolower($className);
		if (!isset(self::$config[$mappingKey])) {
			$mappingKey = 'default_mapping';
		}

		return self::$config[$mappingKey];
	}

	private static function getAnalysis(): array
	{
		if (!self::$config) {
			self::$config = include PIMCORE_PROJECT_ROOT.'/src/AppBundle/Tool/ElasticSearchConfig.php';
		}

		return self::$config['analysis'];
	}

	private static function addFulltextFieldsToMapping(array &$mapping, array $fields, string $lang)
	{
		foreach ($fields as $field) {
			//add field
			if (!isset($mapping['properties'][$field])) {
				$mapping['properties'][$field] = [
					'type' => 'text',
					'term_vector' => 'with_positions_offsets',
				];
			}
			//add folded field
			if (!isset($mapping['properties'][$field]['fields']['folded'])) {
				$mapping['properties'][$field]['fields']['folded'] = [
					'type' => 'text',
					'term_vector' => 'with_positions_offsets',
				];
			}
			//apply hunspell
			$analyzer = ('cs' == $lang || 'sk' == $lang) ? $lang.'_hunspell' : 'standard';
			$mapping['properties'][$field]['analyzer'] = $analyzer;
			$mapping['properties'][$field]['search_analyzer'] = $analyzer;
			$icuAnalyzer = ('cs' == $lang || 'sk' == $lang) ? $lang.'_icu_analyzer' : 'standard';
			$mapping['properties'][$field]['fields']['folded']['analyzer'] = $icuAnalyzer;
			$mapping['properties'][$field]['fields']['folded']['search_analyzer'] = $icuAnalyzer;
		}
	}

	private static function createObjectIndices(): array
	{
		$elasticClient = self::getClient();
		$indices = [];

		$classes = new ClassDefinition\Listing();
		$classes->load();
		foreach ($classes->getClasses() as $class) {
			$className = '\\AppBundle\\Model\\'.$class->getName();
			if (@class_exists($className)) {
				$classImplements = (array) class_implements($className);
				if (!empty($classImplements) && in_array('AppBundle\\Model\\IElasticObject', $classImplements)) {
					$objectMapping = self::getObjectMapping($class->getName());
					foreach ($objectMapping['meta']['languages'] as $lang) {
						$mapping = [
							'_source' => ['enabled' => true],
							'dynamic' => false,
							'properties' => $objectMapping['properties'],
						];
						//apply hunspell on fulltext fields
						if (isset($objectMapping['meta']['fulltextFields'])) {
							self::addFulltextFieldsToMapping($mapping, $objectMapping['meta']['fulltextFields'], $lang);
						}
						$suffix = strtolower($class->getName()).'_'.$lang;
						$indexName = self::getIndex($suffix);
						$elasticClient->indices()->create([
							'index' => $indexName,
							'body' => [
								'settings' => ['analysis' => self::getAnalysis()],
								'mappings' => $mapping,
							]
						]);
						$indices[$suffix] = $indexName;
						self::writeln(sprintf('  <comment>[%s]</comment> <info>✔</info>', $indexName));
					}
				}
			}
		}

		return $indices;
	}

	private static function indexFulltext(array $allowedOnlyClasses = [], bool $copyNotIndexed = true): void
	{
		if (!$allowedOnlyClasses || in_array('Page', $allowedOnlyClasses)) {
			$childList = new Document\Listing();
			$childList->setUnpublished(true);
			$childList->setCondition('type = ?', ['page']);
			self::progressStart('  Document Page', $childList->count());
			$rootDocument = Document::getById(1);
			$queue = [$rootDocument];
			while (!empty($queue)) {
				$document = array_shift($queue);
				if ($document instanceof Document\Page) {
					$document->elasticSearchUpdateFulltext();
					self::progressAdvance();
				}
				$childList->setCondition('(type = ? OR type = ?) AND parentId = ?', ['page', 'folder', $document->getId()]);
				$childList->load();
				foreach ($childList->getItems(0, 0) as $child) {
					$queue[] = $child;
				}
			}
			self::progressEnd();
		} elseif ($copyNotIndexed) {
			$fulltextMapping = self::getObjectMapping('fulltext_mapping');
			self::write('  '.str_pad('Document Page', 20).'<comment>[copy]</comment>');
			foreach ($fulltextMapping['meta']['languages'] as $lang) {
				$suffix = self::DOCUMENT_FULLTEXT_INDEX_KEY.'_'.$lang;
				$oldIndex = self::getIndex($suffix, true);
				if ($oldIndex !== null) {
					$newIndex = self::getIndex($suffix);
					self::copyIndex($oldIndex, $newIndex);
				}
			}
			self::writeln(' <info>✔</info>');
		}

		$classes = new ClassDefinition\Listing();
		$classes->load();
		foreach ($classes->getClasses() as $class) {
			$className = '\\AppBundle\\Model\\'.$class->getName();
			$classListName = $className.'\\Listing';
			if (@class_exists($className) && @class_exists($classListName)) {
				$classImplements = class_implements($className);
				if ($classImplements && in_array(IElasticObjectFulltext::class, $classImplements)) {
					if ($allowedOnlyClasses && !in_array($class->getName(), $allowedOnlyClasses)) {
						if ($copyNotIndexed) {
							self::write(sprintf('  %s<comment>[copy]</comment>', str_pad($class->getName(), 20)));
							$fulltextMapping = self::getObjectMapping('fulltext_mapping');
							foreach ($fulltextMapping['meta']['languages'] as $lang) {
								$suffix = self::OBJECT_FULLTEXT_INDEX_KEY.'_'.$lang;
								$oldIndex = self::getIndex($suffix, true);
								if ($oldIndex !== null) {
									$newIndex = self::getIndex($suffix);
									self::copyIndex($oldIndex, $newIndex, $class->getName());
								}
							}
							self::writeln(' <info>✔</info>');
						}
						continue;
					}
					$list = new $classListName();
					$list->setUnpublished(true);
					$list->setObjectTypes([
						AbstractObject::OBJECT_TYPE_OBJECT,
						AbstractObject::OBJECT_TYPE_VARIANT,
					]);
					$list->load();
					self::progressStart(sprintf('  %s', $class->getName()), $list->count());
					foreach ($list->getItems(0, 0) as $object) {
						$object->elasticSearchUpdateFulltext();
						self::progressAdvance();
					}
					self::progressEnd();
				}
			}
		}
	}

	private static function indexObjects(array $allowedOnlyClasses = [], bool $copyNotIndexed = true): void
	{
		$classes = new ClassDefinition\Listing();
		$classes->load();

		foreach ($classes->getClasses() as $class) {
			$className = '\\AppBundle\\Model\\'.$class->getName();
			$classListName = $className.'\\Listing';
			if (@class_exists($className) && @class_exists($classListName)) {
				$classImplements = class_implements($className);
				if ($classImplements && in_array(IElasticObject::class, $classImplements)) {
					if ($allowedOnlyClasses && !in_array($class->getName(), $allowedOnlyClasses)) {
						if ($copyNotIndexed) {
							self::write(sprintf('  %s<comment>[copy]</comment>', str_pad($class->getName(), 20)));
							$objectMapping = self::getObjectMapping($class->getName());
							foreach ($objectMapping['meta']['languages'] as $lang) {
								$suffix = strtolower($class->getName()).'_'.$lang;
								$oldIndex = self::getIndex($suffix, true);
								if ($oldIndex !== null) {
									$newIndex = self::getIndex($suffix);
									self::copyIndex($oldIndex, $newIndex);
								}
							}
							self::writeln(' <info>✔</info>');
						}
						continue;
					}
					$list = new $classListName();
					$list->setUnpublished(true);
					$list->setObjectTypes([
						AbstractObject::OBJECT_TYPE_OBJECT,
						AbstractObject::OBJECT_TYPE_VARIANT,
					]);
					$list->load();
					self::progressStart(sprintf('  %s', $class->getName()), $list->count());
					foreach ($list->getItems(0, 0) as $object) {
						$object->elasticSearchUpdate();
						self::progressAdvance();
					}
					self::progressEnd();
				}
			}
		}
	}

	private static function copyIndex(string $oldIndex, string $newIndex, string $type = null): void
	{
		$client = self::getClient();
		$body = [
			'source' => ['index' => $oldIndex],
			'dest' => ['index' => $newIndex],
		];
		if ($type !== null) {
			$params['source']['query'] = ['match' => ['type' => $type]];
		}
		$client->reindex(['body' => $body, 'refresh' => true]);
	}

	/**
	 * @param string|bool $fulltext
	 * @param string|bool $objects
	 */
	public static function updateIndicesAndReindexContent($fulltext = false, $objects = false, bool $copyNotReindexed = true): void
	{
		Cache\Runtime::set('elasticSearchUpdateRequest', true);
		Cache\Runtime::set(self::NEW_INDEX_PREFIX_CACHE_KEY, date('Ymdhis'));

		self::writeln('➤ Creating new indices');

		$fulltextIndices = self::createFulltextIndices();
		$objectIndices = self::createObjectIndices();

		if ($objects) {
			self::writeln('➤ Indexing objects');
			$allowedOnlyClasses = [];
			if (is_string($objects)) {
				$allowedOnlyClasses = explode(',', $objects);
			}
			self::indexObjects($allowedOnlyClasses, $copyNotReindexed);
		}

		if ($fulltext) {
			self::writeln('➤ Indexing fulltext');
			$allowedOnlyClasses = [];
			if (is_string($fulltext)) {
				$allowedOnlyClasses = explode(',', $fulltext);
			}
			self::indexFulltext($allowedOnlyClasses, $copyNotReindexed);
		}

		self::write('➤ Switching aliases');
		$elasticClient = self::getClient();
		$aliasPrefix = self::getAliasPrefix();
		foreach ($fulltextIndices + $objectIndices as $suffix => $newIndex) {
			$alias = $aliasPrefix . '_' . $suffix;
			$currentIndex = self::getIndex($suffix, true);
			$elasticClient->indices()->putAlias(['index' => $newIndex, 'name' => $alias]);
			if ($currentIndex !== null) {
				$elasticClient->indices()->delete(['index' => $currentIndex]);
			}
			$elasticClient->indices()->refresh(['index' => $newIndex]);
		}

		Cache\Runtime::set(self::NEW_INDEX_PREFIX_CACHE_KEY, null);
		Cache\Runtime::set('elasticSearchUpdateRequest', null);
		self::writeln(' <info>✔</info>');
	}
}
