<?php

namespace Website\Form;

class Base extends \Zend_Form
{
	protected $fileUploads = [];

	private static $emailValidator = null;
	private static $notEmptyValidator = null;
	private static $phoneValidator = null;
	private static $numberValidator = null;
	private static $dateValidator = null;
	private static $passwordLengthValidator = null;
	private static $passwordIdenticalValidator = null;
	private static $urlValidator = null;
	private static $floatValidator = null;
	private static $imageValidator = null;
	private static $fileSizeValidator = null;
	private static $fileUploadedValidator = null;
	private static $onlyOneFileValidator = null;
	private static $zipCodeValidator = null;

	public function init()
	{
		parent::init();

		$this->setMethod(\Zend_Form::METHOD_POST);
		$this->setEnctype(\Zend_Form::ENCTYPE_URLENCODED);
		$this->setName('base-form');
	}

	//lazy loaded static validators
	public function __get($name)
	{
		if (property_exists('Website\Form\Base', $name)) {
			return self::getValidator($name);
		}

		return parent::__get($name);
	}
	private static function getValidator($name)
	{
		if (self::$$name != null) return self::$$name;

		switch ($name) {
			case 'notEmptyValidator': {
				self::$notEmptyValidator = new \Zend_Validate_NotEmpty();
				self::$notEmptyValidator->setMessage('msg_empty_field');
			} break;
			case 'emailValidator': {
				$emailHostnameValidator = new \Zend_Validate_Hostname(\Zend_Validate_Hostname::ALLOW_DNS);
				$emailHostnameValidator->setMessage('msg_invalid_email');
				self::$emailValidator = new \Zend_Validate_EmailAddress(['hostname' => $emailHostnameValidator]);
				self::$emailValidator->setMessage('msg_invalid_email');
			} break;
			case 'phoneValidator': {
				self::$phoneValidator = new \Zend_Validate_Regex('/^(\+420|\+421|00420|00421)?[ ]*[0-9]{3}[ ]*[0-9]{3}[ ]*[0-9]{3}$/');
				self::$phoneValidator->setMessage('msg_bad_format_phone');
			} break;
			case 'numberValidator': {
				self::$numberValidator = new \Zend_Validate_Digits();
				self::$numberValidator->setMessage('msg_bad_format');
			} break;
			case 'floatValidator': {
				self::$floatValidator = new \Zend_Validate_Float(\Zend_Registry::get('Zend_Locale'));
				self::$floatValidator->setMessage('msg_bad_format');
			} break;
			case 'passwordLengthValidator': {
				self::$passwordLengthValidator = new \Zend_Validate_StringLength(5, 32);
				self::$passwordLengthValidator->setMessage('msg_bad_format');
			} break;
			case 'passwordIdenticalValidator': {
				self::$passwordIdenticalValidator = new \Zend_Validate_Identical('password');
				self::$passwordIdenticalValidator->setMessage('msg_passwords_do_not_match');
			} break;
			case 'dateValidator': {
				self::$dateValidator = new \Zend_Validate_Date();
				self::$dateValidator->setMessage('msg_bad_format');
			} break;
			case 'urlValidator': {
				self::$urlValidator = new \Zend_Validate_Callback(['Zend_Uri', 'check']);
				self::$urlValidator->setMessage('msg_bad_format');
			} break;
			case 'imageValidator': {
				self::$imageValidator = new \Zend_Validate_File_Extension(['jpg', 'jpeg', 'gif', 'png']);
				self::$imageValidator->setMessage('msg_not_an_image');
			} break;
			case 'fileSizeValidator': {
				self::$fileSizeValidator = new \Zend_Validate_File_FilesSize(\Website\Model\FileManager::MAX_FILE_SIZE);
				self::$fileSizeValidator->setMessage('msg_file_too_big');
			} break;
			case 'fileUploadedValidator': {
				self::$fileUploadedValidator = new \Zend_Validate_File_Upload();
				self::$fileUploadedValidator->setMessage('msg_file_not_uploaded');
				self::$fileUploadedValidator->setMessage('msg_file_too_big', 'fileUploadErrorIniSize');
				self::$fileUploadedValidator->setMessage('msg_file_too_big', 'fileUploadErrorFormSize');
				self::$fileUploadedValidator->setMessage('msg_empty_field', 'fileUploadErrorNoFile');
			} break;
			case 'onlyOneFileValidator': {
				self::$onlyOneFileValidator = new \Zend_Validate_File_Count(1);
				self::$onlyOneFileValidator->setMessage('msg_file_not_uploaded');
			} break;
			case 'zipCodeValidator': {
				self::$zipCodeValidator = new \Zend_Validate_Regex('/^[0-9]{3} ?[0-9]{2}$/');
				self::$zipCodeValidator->setMessage('msg_bad_format');
			} break;
		}

		return self::$$name;
	}

	/**
	 * 
	 * @param mixed $objectOrClassName
	 */
	public function initFileUploads($objectOrClassName)
	{
		if ($objectOrClassName instanceof \Pimcore\Model\Object\Concrete) {
			$fieldDefinitions = $objectOrClassName->getClass()->getFieldDefinitions();
		} else {
			$fieldDefinitions = \Pimcore\Model\Object\ClassDefinition::getByName($objectOrClassName)->getFieldDefinitions();
		}
		foreach ($fieldDefinitions as $fieldDefinition) {
			//upload elements are identified by the prefix "upload"
			if (substr($fieldDefinition->name, 0, 6) == 'upload') {
				if (in_array($fieldDefinition->fieldtype, ['href', 'multihref', 'image'])) {
					$data = ($objectOrClassName instanceof \Pimcore\Model\Object\Concrete)
						? $objectOrClassName->{'get'.ucfirst($fieldDefinition->name)}()
						: null;
					$fileUpload = $this->addFileUpload($fieldDefinition, $data);
					$this->fileUploads[$fileUpload['fieldName']] = $fileUpload;
				}
			}
		}
		if (!empty($this->fileUploads)) {
			$this->setEnctype(self::ENCTYPE_MULTIPART);
		}
	}

	private function addFileUpload(\Pimcore\Model\Object\ClassDefinition\Data $fieldDefinition, $data = null)
	{
		//do the configuration
		$requiredParts = 0;
		$maxParts = 1;
		$uploadFolderPath = ($fieldDefinition->fieldtype == 'href' || $fieldDefinition->fieldtype == 'multihref') 
			? $fieldDefinition->assetUploadPath
			: $fieldDefinition->uploadPath;

		if (($fieldDefinition->fieldtype == 'href' || $fieldDefinition->fieldtype == 'image') && $fieldDefinition->mandatory) $requiredParts = 1;
		if ($fieldDefinition->fieldtype == 'multihref') $maxParts = $fieldDefinition->maxItems;
		$configuration = [
			'fieldName' => $fieldDefinition->name,
			'fieldType' => $fieldDefinition->fieldtype,
			'maxParts' => $maxParts,
			'requiredParts' => $requiredParts,
			'folder' => $uploadFolderPath,
			'subFolder' => null
		];
		if (isset($this->fileUploads[$fieldDefinition->name])) {
			$configuration = array_merge($configuration, $this->fileUploads[$fieldDefinition->name]);
		}
		//init the inputs
		for ($i = 1; $i <= $configuration['maxParts']; $i++) {
			$forceEdit = 0;
			$assetId = null;

			if ($fieldDefinition->fieldtype == 'multihref') {
				if (isset($data[$i - 1]) && $data[$i - 1] instanceof Asset) $assetId = $data[$i - 1]->getId();
			} else {
				if ($data instanceof Asset) $assetId = $data->getId();
			}
			if (!$assetId && $i <= $configuration['requiredParts']) $forceEdit = 1;

			$getter = $fieldDefinition->name.$i;
			$edit = new \Zend_Form_Element_Hidden('edit_'.$getter);
			$edit->setRequired(false);
			$edit->setValue($forceEdit);
			$this->addElement($edit);
			$this->getElement('edit_'.$getter)->setDecorators(['ViewHelper']);
			if ($assetId) {
				$this->getElement('edit_'.$getter)->setAttrib('data-id', $assetId);
			}

			$remove = new \Zend_Form_Element_Hidden('remove_'.$getter);
			$remove->setRequired(false);
			$remove->setValue(0);
			$this->addElement($remove);
			$this->getElement('remove_'.$getter)->setDecorators(['ViewHelper']);

			$translator = $this->getTranslator();
			$uploadElement = new \Zend_Form_Element_File($getter);
			$uploadElement->setRequired(false)
				->setMaxFileSize(\Website\Model\FileManager::MAX_FILE_SIZE)
				->setAttrib('data-required', (int)($configuration['requiredParts'] >= $i))
				->setAttrib('data-maxParts', $configuration['maxParts'])
				->setAttrib('data-label', ($translator) ? $translator->translate('label_'.$fieldDefinition->name) : 'label_'.$fieldDefinition->name);
			$this->addElement($uploadElement, $getter);
			if ($configuration['requiredParts'] >= $i) {
				$this->getElement($getter)->setAttrib('required', 'required');
			}
			$this->getElement($getter)->setDecorators(['File']);

		}

		return $configuration;
	}

	public function handleFileUploads(\Pimcore\Model\Object\Concrete &$object, $fieldName = null)
	{
		$values = $this->getValues();
		$fileManager = new \Website\Model\FileManager();
		$result = true;
		$translator = null;

		foreach ($this->fileUploads as $fileUpload) {
			if ($fieldName != null && $fileUpload['fieldName'] != $fieldName) continue;

			$getter = 'get'.ucfirst($fileUpload['fieldName']);

			for ($i = 1; $i <= $fileUpload['maxParts']; $i++) {
				$key = $fileUpload['fieldName'].$i;
				$translator = $this->$key->getTranslator();
				if ($values['edit_'.$key]) { //add / replace
					if (!$this->$key->receive() || !$this->$key->isUploaded()) {
						$this->$key->addError(($translator) ? $translator->translate('msg_file_not_uploaded') : 'msg_file_not_uploaded');
						$result = false;
						continue;
					}
					if (!$fileManager->uploadFile($this->$key, $object, $fileUpload['fieldName'], $i, $fileUpload['folder'], $fileUpload['subFolder'])) {
						$this->$key->addError(($translator) ? $translator->translate('msg_file_not_uploaded') : 'msg_file_not_uploaded');
						$result = false;
					}
				} elseif($i > $fileUpload['requiredParts'] && $values['remove_'.$key]) { // maybe remove
					$asset = null;
					$data = $object->$getter();
					if ($fileUpload['fieldType'] == 'href' || $fileUpload['fieldType'] == 'image') $asset = $data;
					elseif ($fileUpload['fieldType'] == 'multihref') $asset = (is_array($data) && isset($data[$i - 1])) ? $data[$i - 1] : null;
					if ($asset instanceof Asset) { // now remove
						try {
							$asset->delete();
						} catch (\Exception $e) {
							\Pimcore\Log\Simple::log('exceptions', 'REMOVING FILE - '.$e->getMessage()."\n".$e->getTraceAsString());
							$this->$key->addError('msg_file_not_removed');
							$result = false;
						}
					}
				}
			}
		}

		return $result;
	}

	protected function setUpJsValidation()
	{
		$validate = false;
		foreach ($this->getElements() as $element) {
			$translator = $element->getTranslator();
			$messages = [];
			$validationAttribs = [];
			$validators = $element->getValidators();
			$conditionalValidator = null;
			//for now there is only max one conditional validator support
			if (array_key_exists('Website\Form\Validator\FieldDepends', $validators)) {
				$conditionalValidator = $validators['Website\Form\Validator\FieldDepends'];
			}
			foreach ($validators as $name => $validator) {
				if ($name == 'Website\Form\Validator\FieldDepends') continue;
				$msgTemplate = current($validator->getMessageTemplates());
				$msg = ($translator) ? $translator->translate($msgTemplate) : $msgTemplate;
				if ($validator instanceof \Zend_Validate_NotEmpty) {
					$validationAttribs['required'] = 'required';
					$messages['required'] = $msg;
				} elseif ($validator instanceof \Zend_Validate_Digits) {
					$validationAttribs['digits'] = 'digits';
					$messages['digits'] = $msg;
				} elseif ($validator instanceof \Zend_Validate_EmailAddress) {
					$validationAttribs['email'] = 'email';
					$messages['email'] = $msg;
				} elseif ($validator instanceof \Zend_Validate_Identical) {
					$validationAttribs['equalTo'] = 'form#'.$this->getId().' #'.$validator->getToken();
					$messages['equalTo'] = $msg;
				} elseif ($validator instanceof \Zend_Validate_Regex) {
					$pattern  = $validator->getPattern();
					$validationAttribs['pattern'] = substr($pattern, 1, strlen($pattern) - 2);
					$messages['pattern'] = $msg;
				}
			}
			if (!$conditionalValidator) {
				foreach ($validationAttribs as $key => $value) {
					$element->setAttrib($key, $value);
				}
			} else {
				$messages['required'] = ($translator) ? $translator->translate('msg_empty_field') : 'msg_empty_field';
				$depends = [
					'key' => $conditionalValidator->getContextKey(),
					'value' => $conditionalValidator->getTestValue()
				];
				if (!empty($validationAttribs)) {
					$depends['additionalValidators'] = $validationAttribs;
				}
				$element->setAttrib('depends', json_encode($depends));
			}
			if (!empty($messages)) {
				$validate = true;
				$element->setAttrib('data-error-messages', json_encode($messages));
			}
		}
		//file uploader
		foreach ($this->fileUploads as $fileUpload) {
			//required files
			for ($i = 1; $i <= $fileUpload['requiredParts']; $i++) {
				$validate = true;
				$messages = [];
				$translator = $this->{$fileUpload['fieldName'].$i}->getTranslator();
				$this->{$fileUpload['fieldName'].$i}->setAttrib('required', 'required');
				$messages['required'] = ($translator) ? $translator->translate('msg_select_file') : 'msg_select_file';
				if ($fileUpload['fieldType'] == 'image') {
					$mimeTypes = [];
					foreach ($this->imageValidator->getExtension() as $extension) {
						$mimeTypes[] = 'image/'.$extension;
					}
					$this->{$fileUpload['fieldName'].$i}->setAttrib('accept', implode(',', $mimeTypes));
					$messages['accept'] = ($translator) ? $translator->translate('msg_not_an_image') : 'msg_not_an_image';
				}
				$this->{$fileUpload['fieldName'].$i}->setAttrib('data-error-messages', json_encode($messages));
			}
			//conditionally required files
			for ($i = $fileUpload['requiredParts'] + 1; $i <= $fileUpload['maxParts']; $i++) {
				$validate = true;
				$messages = [];
				$translator = $this->{$fileUpload['fieldName'].$i}->getTranslator();
				$messages['required'] = ($translator) ? $translator->translate('msg_select_file') : 'msg_select_file';
				$depends = [
					'key' => 'edit_'.$fileUpload['fieldName'].$i,
					'value' => '1' 
				];
				$this->{$fileUpload['fieldName'].$i}->setAttrib('depends', json_encode($depends));
				$this->{$fileUpload['fieldName'].$i}->setAttrib('data-error-messages', json_encode($messages));
			}
			
		}
		$this->setAttrib('js-validation', $validate);
	}

	public function isValid($data)
	{
		//cycle through the elements and remove the additional validators
		//on elements with conditional validator if the condition is not met
		foreach ($this->getElements() as $element) {
			$validators = $element->getValidators();
			if (array_key_exists('Website\Form\Validator\FieldDepends', $validators)
				&& $data[$validators['Website\Form\Validator\FieldDepends']->getContextKey()] != $validators['Website\Form\Validator\FieldDepends']->getTestValue()
			) {
				foreach (array_keys($validators) as $key) {
					if ($key != 'Website\Form\Validator\FieldDepends') {
						$element->removeValidator($key);
					}
				}
			}
		}
		//add file uploads validators
		$removedElements = [];
		foreach ($this->fileUploads as $fileUpload)  {
			for ($i = 1; $i <= $fileUpload['maxParts']; $i++) {
				$getter = $fileUpload['fieldName'].$i;
				if (!empty($data['edit_'.$getter]) && isset($this->$getter)) {
					$this->$getter->setRequired(true);
					$this->$getter->addValidator($this->fileSizeValidator);
					$this->$getter->addValidator($this->fileUploadedValidator);
					//$this->$getter->addValidator($this->onlyOneFileValidator);
					if ($fileUpload['fieldType'] == 'image') {
						$this->$getter->addValidator($this->imageValidator);
					}
				} else { # ajax + file upload + zend bug hack (remove empty elements from form during validation)
					$removedElements[] = $this->getElement($getter);
					$this->removeElement($getter);
				}
			}
		}

		$isValid = parent::isValid($data);

		foreach($removedElements as $removedElement) {
			$this->addElement($removedElement);
		}

		if (!$isValid) {
			\Pimcore\Log\Simple::log('forms', $this->getName() . ' - ' . serialize($this->getMessages()));
		}

		return $isValid;
	}
}
