<?php

/**
 * simple class to help with elastic searching
 * 
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */

namespace Website\Model;

class ElasticSearch
{
	private static $instances;

	/**
	 * @var \Elasticsearch\Client
	 */
	protected $elasticClient = null;
	protected $index = null;
	protected $language = 'cs';
	protected $urlPrefix = null;

	/**
	 * @param string $language
	 * @param string $urlPrefix
	 * @return ElasticSearch
	 */
	public static function getInstance($language, $urlPrefix = null)
	{
		$key = $language.$urlPrefix;
		if (!isset(self::$instances[$key])) {
			self::$instances[$key] = new ElasticSearch($language, $urlPrefix);
		}
		return self::$instances[$key];
	}

	private function __construct($language, $urlPrefix = null)
	{
		$this->language = $language;
		$this->elasticClient = \Website\Tool\ElasticSearch::getClient();
		$this->index = \Website\Tool\ElasticSearch::getMainIndexName(false);
		$this->urlPrefix = $urlPrefix;

		if (!$this->elasticClient->indices()->existsAlias(['name' => $this->index])) {
			throw new \Exception('Elastic search main index not found.');
		}
	}

	public function fulltextSearch($filter = [])
	{
		// query base
		$baseOptions = ['type' => [
			\Website\Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY . '_' . $this->language,
			\Website\Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY . '_' . $this->language
		]];
		if (isset($filter['query']) && !empty($filter['query'])) {
			$baseOptions['query']['multi_match']['query'] = $filter['query'];
			$baseOptions['query']['multi_match']['fields'] = ['title^4', 'keywords^3', 'description^2', 'content'];
		}
		$filteredQuery = $this->queryBase($baseOptions, $filter);

		$aggregations = [];
		$aggregations['type'] = $this->fulltextTypeAggregation($filteredQuery);

		if (isset($filter['type']) && !empty($filter['type'])) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['type'] = $filter['type'];
		}

		$pagination = $this->query($filteredQuery, $filter, true, true, true);
		$pagination->aggregations = $aggregations;

		return $pagination;
	}

	private function fulltextTypeAggregation($query)
	{
		//brands
		$aggQuery = [];
		$aggQuery['type']['terms']['size'] = 100;
		$aggQuery['type']['terms']['field'] = 'type';

		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		$types = [];
		foreach ($result['aggregations']['type']['buckets'] as $bucket) {
			$types[$bucket['key']] = $bucket['doc_count'];
		}

		return $types;
	}

	private function queryBase($options = [], $filter = [])
	{
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = $options['type'];

		// set query for filtered query
		$query = (isset($options['query'])) ? $options['query'] : null;
		if (is_array($query)) {
			$filteredQuery['body']['query']['filtered']['query'] = $query;
		} elseif ($query == 'random_score') {
			$filteredQuery['body']['query']['filtered']['query']['function_score']['functions'][0]['random_score'] = new \stdClass();
		} elseif(isset($filter['ids_score']) && isset($filter['ids']) && $filter['ids']) {
			$functionScore = [
				'boost_mode' => 'replace',
				'script_score' => [
					'script' => "count = ids.size(); return -ids.indexOf(Integer.parseInt(doc['id'].value));",
					'params' => [
						'ids' => $filter['ids']
					]
				]
			];
			$filteredQuery['body']['query']['filtered']['query']['function_score'] = $functionScore;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];
		}

		// add filters
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['published'] = true;

		return $filteredQuery;
	}

	private function constructSimplePagination($resultSet, $limit, $page, $totalCount)
	{
		$pagination = new \stdClass();
		$pagination->current = $page;
		$pagination->pageCount = ($limit) ? (int) ceil($totalCount / $limit) : 1;
		if ($page > 1) {
			$pagination->previous = $page - 1;
		}
		if ($page < $pagination->pageCount) {
			$pagination->next = $page + 1;
		}
		$pagination->totalCount = $totalCount;
		$pagination->first = 1;
		$pagination->last = $pagination->pageCount;
		$pagination->items = empty($resultSet) ? [] : $resultSet['hits']['hits'];
		$pagination->urlprefix = $this->urlPrefix;

		return $pagination;
	}

	private function query($filteredQuery, $filter, $foldedFallback = true, $paginate = true, $highlighting = false)
	{
		//add folded fallback fields
		$fulltextFields = [];
		if ($foldedFallback && isset($filter['query']) && !empty($filter['query'])) {
			$fulltextFields = $filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'];
			$fields = [];
			foreach($fulltextFields as $field) {
				$parts = explode('^', $field);
				$foldedField = $parts[0] . '.folded';
				if (isset($parts[1])) $foldedField .= '^' . $parts[1];
				$fields[] = $field;
				$fields[] = $foldedField;
			}
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = $fields;
		}

		//get total count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];

		//sort, only when no searchQuery is set and some results matched and some sort is defined...
		if ($totalCount && (!isset($filter['query']) || empty($filter['query'])) &&
		isset($filter['sort']) && $filter['sort'] && !isset($filter['ids_score'])) {
			$sortStruct = $filter['sort'];
			if (is_array($sortStruct)) {
				foreach($sortStruct as $key => $sort) {
					$filteredQuery['body']['sort'][][$key] = $sort;
				}
			} elseif (stristr($sortStruct, ':')) {
				$gps = explode(':', $sortStruct);
				$filteredQuery['body']['sort'][]['_geo_distance'] = [
					'location' => str_replace(':', ',', $sortStruct),
					'order' => 'asc',
					'unit' => 'km',
					'distance_type' => 'plane'
				];
			} else {
				//TODO define sorting types
			}
		}

		//add highlighting
		if ($highlighting && $fulltextFields && isset($filter['query']) && !empty($filter['query'])) {
			$highlight = ['fields' => []];
			foreach($fulltextFields as $field) {
				$parts = explode('^', $field);
				$highlight['fields'][$parts[0]] = [
					'matched_fields' => [$parts[0]],
					'type' => 'fvh'
				];
				if ($foldedFallback) {
					$highlight['fields'][$parts[0]]['matched_fields'][] = $parts[0] . '.folded';
				}
			}
			$filteredQuery['body']['highlight'] = $highlight;
		}

		//pagination
		$limit = isset($filter['limit']) ? intval($filter['limit']) : 999;
		if ($paginate && $limit) {
			$page = isset($filter['page']) ? intval($filter['page']) : 1;
			if ((int) $page > ceil($totalCount / $limit))
				$page = ceil($totalCount / $limit);
			if ($page < 1) $page = 1;
			$filteredQuery['body']['from'] = isset($filter['offset']) ? ($page - 1) * $limit : 0;
			$filteredQuery['body']['size'] = isset($filter['offset']) ? $limit : $page * $limit;
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		if ($paginate) {
			return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
		} else {
			return $resultSet;
		}
	}

}
