<?php 
set_time_limit(60000);
include_once(__DIR__."/../../pimcore/cli/startup.php");
//execute in admin mode
define("PIMCORE_ADMIN", true);
//disable cache and verrsions
\Pimcore\Model\Cache::disable();
\Pimcore\Model\Version::disable();

const DOCUMENT_ACTIONS = 'DOCUMENT ROUTED ACTIONS';
const FORM_ACTIONS = 'FORM HANDLERS';

$availableCommands = [
	'class' => [
		'create' => ['name' => true, 'localized' => false, 'icon' => false, 'nomap' => false],
		'delete' => ['name' => true]
	],
	'classmap' => [
		'create' => ['name' => true, 'nolist' => false],
		'delete' => ['name' => true]
	],
	'action' => [
		'document' => ['name' => true, 'controller' => false, 'form' => false, 'noview' => false, 'view' => false],
		'formhandler' => ['name' => true, 'controller' => false],
	],
	'document' => [
		'page' => ['name' => false, 'controller' => false, 'action' => false, 'parent' => false, 'view' => false, 'master' => false],
		'email' => ['name' => true, 'parent' => false]
	],
	'area' => [
		'create' => ['name' => true, 'icon' => false],
		'delete' => ['name' => true]
	],
	'thumb' => [
		'create' => ['name' => true, 'method' => true, 'width' => false, 'height' => false],
		'delete' => ['name' => true]
	]
];

$subcommandsList = [];
foreach ($availableCommands as $command => $subcommands) {
	$subcommandsList[] = sprintf('%s [%s]', $command, implode(', ', array_keys($subcommands)));
}
try {
    $opts = new \Zend_Console_Getopt([
		'verbose|v' => 'show detailed information during the maintenance (for debug, ...)',
		'command|c=w' => sprintf('command, available commands: %s', implode(', ', array_keys($availableCommands))),
		'subcommand|s=w' => sprintf('available subcommands: %s', implode(', ', $subcommandsList)),
		'name|n=s' => 'name/title',
		'controller=w' => 'controller name',
		'action=w' => 'action name',
		'view=s' => 'path to view script',
		'parent=s' => 'parent document path or id',
		'master=s' => 'master document path or id',
		'form=w' => 'form name',
		'method=w' => 'thumbnail method',
		'width=w' => 'width in pixels',
		'height=w' => 'height in pixels',
		'icon=s' => 'relative path to an image',
		'localized' => 'should it be localized?',
		'nomap' => 'do not create class mapping',
		'nolist' => 'do not include list in mapping',
		'noview' => 'do not create view script',
        'help|h' => 'show this help'
    ]);
	$opts->parse();
} catch (\Exception $e) {
    echo $e->getMessage();
	exit;
}

// display help message
if($opts->getOption("help")) {
    echo $opts->getUsageMessage();
    exit;
}

if($opts->getOption("verbose")) {
    $writer = new \Zend_Log_Writer_Stream('php://output');
    $logger = new \Zend_Log($writer);
    \Logger::addLogger($logger);

    // set all priorities
    \Logger::setPriorities([
        \Zend_Log::DEBUG,
        \Zend_Log::INFO,
        \Zend_Log::NOTICE,
        \Zend_Log::WARN,
        \Zend_Log::ERR,
        \Zend_Log::CRIT,
        \Zend_Log::ALERT,
        \Zend_Log::EMERG
    ]);
}

fputs(STDOUT, "\n");

# command syntax check
$command = $opts->getOption('command');
if (!array_key_exists($command, $availableCommands)) {
	fputs(STDOUT, sprintf("Error: command [%s] not available, available commands: [%s].\n", $command, implode('|', array_keys($availableCommands))));
	exit;
}
$action = $opts->getOption('subcommand');
if (!array_key_exists($action, $availableCommands[$command])) {
	fputs(STDOUT, sprintf("Error: subcommand [%s] for command [%s] not available, available subcommands: [%s].\n", $action, $command, implode('|', array_keys($availableCommands[$command]))));
	exit;
}
foreach ($availableCommands[$command][$action] as $key => $required) {
	if (!$required) continue;

	$param = $opts->getOption($key);
	if (!$param) {
		fputs(STDOUT, sprintf("Error: missing required param [%s]\n", $key));
		exit;
	}
}

#execute the right command with right parameters
switch($command.':'.$action) {
	case 'class:create': {
		$name = $opts->getOption('name');
		$localized = $opts->getOption('localized');
		$icon = normalizePathParam($opts->getOption('icon'));
		$id = createClassDefinition($name, $localized, $icon);
		$nomap = $opts->getOption('nomap');
		fputs(STDOUT, sprintf("Class [%s] created with ID %d\n", $name, $id));
		if (!$nomap) {
			createClassMapping($id);
			fputs(STDOUT, sprintf("Class [%s] mapped\n", $name));
		}
	} break;
	case 'class:delete': {
		$name = $opts->getOption('name');
		deleteClassMapping($name);
		fputs(STDOUT, sprintf("Classmapping for class [%s] deleted\n", $name));
		deleteClassDefinition($name);
		fputs(STDOUT, sprintf("Class [%s] deleted\n", $name));
	} break;
	case 'classmap:create': {
		$name = $opts->getOption('name');
		$noList = $opts->getOption('nolist');
		createClassMapping($name, !$noList);
		fputs(STDOUT, sprintf("Class [%s] mapped\n", $name));
	} break;
	case 'classmap:delete': {
		$name = $opts->getOption('name');
		deleteClassMapping($name);
		fputs(STDOUT, sprintf("Classmapping for class [%s] deleted\n", $name));
	} break;
	case 'action:document': {
		$name = $opts->getOption('name');
		if ($name) $name = lcfirst($name);
		$controller = $opts->getOption('controller');
		if ($controller) $controller = ucfirst($controller);
		$form = $opts->getOption('form');
		if ($form) $form = ucfirst($form);
		$noView = $opts->getOption('noview');
		$view = $opts->getOption('view');
		if ($view) $view = normalizePathParam($view);
		createAction($name, $controller, $form, ($view) ? $view : !$noView);
		fputs(STDOUT, sprintf("Action [%sAction] created in [%sController]\n", $name, ($controller) ? $controller : 'Cms'));
		if ($form) {
			createFormHandler($form, $controller);
			fputs(STDOUT, sprintf("Form handler [handle%sForm] created in [%sController]\n", $form, ($controller) ? $controller : 'Cms'));
		}
	} break;
	case 'action:formhandler': {
		$name = $opts->getOption('name');
		if ($name) $name = ucfirst($name);
		$controller = $opts->getOption('controller');
		if ($controller) $controller = ucfirst($controller);
		createFormHandler($name, $controller);
		fputs(STDOUT, sprintf("Form handler [handle%sForm] created in [%sController]\n", $name, ($controller) ? $controller : 'Cms'));
	} break;
	case 'document:page': {
		$name = $opts->getOption('name');
		$controller = $opts->getOption('controller');
		$action = $opts->getOption('action');
		$parent = $opts->getOption('parent');
		$view = $opts->getOption('view');
		if ($view) $view = normalizePathParam($view);
		$master = $opts->getOption('master');
		$document = createDocumentPage($name, $controller, $action, $parent, $view, $master);
		fputs(STDOUT, sprintf("Document [%s] created with ID [%d]\n", $document->getRealFullPath(), $document->getId()));
	} break;
	case 'document:email': {
		$name = $opts->getOption('name');
		$parent = $opts->getOption('parent');
		$email = createDocumentEmail($name, $parent);
		fputs(STDOUT, sprintf("Email [%s] created with ID [%d]\n", $email->getRealFullPath(), $email->getId()));
	} break;
	case 'area:create': {
		$name = $opts->getOption('name');
		if ($name) $name = strtolower($name);
		$icon = normalizePathParam($opts->getOption('icon'));
		createArea($name, $icon);
		fputs(STDOUT, sprintf("Area [%s] created\n", $name));
	} break;
	case 'area:delete': {
		$name = $opts->getOption('name');
		if ($name) $name = strtolower($name);
		deleteArea($name);
		fputs(STDOUT, sprintf("Area [%s] deleted\n", $name));
	} break;
	case 'thumb:create': {
		$name = $opts->getOption('name');
		$method = $opts->getOption('method');
		$width = $opts->getOption('width');
		$height = $opts->getOption('height');
		createThumbnailPipeline($name, $method, $width, $height);
		fputs(STDOUT, sprintf("Thumbnail [%s] created\n", $name));
	} break;
	case 'thumb:delete': {
		$name = $opts->getOption('name');
		deleteThumbnailPipeline($name);
		fputs(STDOUT, sprintf("Thumbnail [%s] deleted\n", $name));
	} break;
	default: {
		fputs(STDOUT, sprintf("Error: [%s:%s] not implemented yet\n", $command, $action));
		exit;
	} break;
}
exit;


/*					HELPER FUNCTIONS						*/


function createClassDefinition($name, $localized = false, $icon = null)
{
	$classDefinition = new \Pimcore\Model\Object\ClassDefinition();
	$classDefinition->setName($name);
	$classDefinition->setCreationDate(time());
	$classDefinition->setModificationDate(time());
	$classDefinition->setUserOwner(2);
	$classDefinition->setUserModification(2);
	//$classDefinition->setUseTraits(false);
	$classDefinition->setAllowInherit(false);
	$classDefinition->setAllowVariants(false);
	$classDefinition->setShowVariants(false);
	if($icon) $classDefinition->setIcon((string)$icon);
	$classDefinition->setPreviewUrl('');
	$classDefinition->setPropertyVisibility([
		'grid' => [
			'id' => true,
			'path' => false,
			'published' => true,
			'modificationDate' => false,
			'creationDate' => true 
		],
		'search' => [
			'id' => true,
			'path' => true,
			'published' => true,
			'modificationDate' => true,
			'creationDate' => true
		]
	]);

	$fieldDefinitions = [];
	$nameInput = new \Pimcore\Model\Object\ClassDefinition\Data\Input();
	$nameInput->setWidth(600);
	$nameInput->setName('name');
	$nameInput->setTitle('Name');
	$nameInput->setMandatory(false);
	$nameInput->setVisibleGridView(true);
	$nameInput->setVisibleSearch(true);
	if ($localized) {
		$lf = new \Pimcore\Model\Object\ClassDefinition\Data\Localizedfields();
		$lf->setName('localizedfields');
		$lf->addChild($nameInput);
		$fieldDefinitions['localizedfields'] = $lf;
	} else {
		$fieldDefinitions['name'] = $nameInput;
	}
	$classDefinition->setFieldDefinitions($fieldDefinitions);

	$panel = new \Pimcore\Model\Object\ClassDefinition\Layout\Panel();
	$panel->setLabelWidth(100);
	$panel->setName($name);
	$panel->setChilds(array_values($fieldDefinitions));
	$layout = new \Pimcore\Model\Object\ClassDefinition\Layout\Panel();
	$layout->setLabelWidth(100);
	$layout->setName('pimcore_root');
	$layout->addChild($panel);

	$classDefinition->setLayoutDefinitions($layout);

	try {
		$classDefinition->save();
		return $classDefinition->getId();
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: %s\n", $e->getMessage()));
		exit;
	}
}

function deleteClassDefinition($name)
{
	$classDefinition = \Pimcore\Model\Object\ClassDefinition::getByName($name);
	if (!$classDefinition) {
		fputs(STDOUT, sprintf("Error: class [%s] not found\n", $name));
		exit;
	}

	try {
		$classDefinition->delete();
		return true;
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: %s\n", $e->getMessage()));
		exit;
	}
}

function createClassMapping($classNameOrId, $mapList = true)
{
	$className = getClassName($classNameOrId);
	$classMap = loadXmlConfig(PIMCORE_WEBSITE_VAR . "/config/classmap.xml");

	$parent = sprintf('Object_%s', $className);
	$parentList = sprintf('Object_%s_Listing', $className);
	$child = sprintf('Website\\Model\\%s', $className);
	$childList = sprintf('Website\\Model\\%s\\Listing', $className);
	if (!isset($classMap->$parent)) {
		$classMap->$parent = $child;
	}
	createMappedClass($className);
	if ($mapList) {
		if (!isset($classMap->$parentList)) {
			$classMap->$parentList = $childList;
		}
		createMappedClassList($className);
	}

	return writeXmlConfig($classMap, PIMCORE_WEBSITE_VAR . "/config/classmap.xml");
}

function deleteClassMapping($classNameOrId)
{
	$className = getClassName($classNameOrId);
	$classMap = loadXmlConfig(PIMCORE_WEBSITE_VAR . "/config/classmap.xml");

	$parent = sprintf('Object_%s', $className);
	$parentList = sprintf('Object_%s_Listing', $className);
	if (isset($classMap->$parent)) {
		unset($classMap->$parent);
		deleteMappedClass($className);
	}
	if (isset($classMap->$parentList)) {
		unset($classMap->$parentList);
		deleteMappedClassList($className);
	}

	return writeXmlConfig($classMap, PIMCORE_WEBSITE_VAR . "/config/classmap.xml");
}

function getClassName($classNameOrId)
{
	$class = null;
	$className = null;
	if (is_int($classNameOrId)) {
		$class = \Pimcore\Model\Object\ClassDefinition::getById($classNameOrId);
	} else {
		$class = Pimcore\Model\Object\ClassDefinition::getByName($classNameOrId);
	}

	if ($class) $className = $class->getName();

	if (!$className) {
		fputs(STDOUT, sprintf("Error: class [%s] does not exist\n", $classNameOrId));
		exit;
	}

	return $className;
}

function loadXmlConfig($pathOrXmlString = null)
{
	if ($pathOrXmlString === null) {
$pathOrXmlString = <<<EOT
<?xml version="1.0"?>
<zend-config xmlns:zf="http://framework.zend.com/xml/zend-config-xml/1.0/">
</zend-config>
EOT;
	}
	$config = null;
	try {
		$config = new \Zend_Config_Xml($pathOrXmlString , null, ["allowModifications" => true]);
	} catch (Exception $e) {}
	if (!$config instanceof \Zend_Config) {
		fputs(STDOUT, sprintf("Error: could not load [%s]\n", $pathOrXmlString));
		exit;
	}

	return $config;
}

function writeXmlConfig($config, $path)
{
	try {
		$writer = new \Zend_Config_Writer_Xml([
			"config" => $config,
			"filename" => $path
		]);
		$writer->write();
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not write [%s]\n", $path));
		exit;
	}

	return true;
}

function normalizeScriptFileName($action)
{
	return strtolower(preg_replace('/([A-Z]+)/', "-$1", lcfirst($action)));
}

function normalizePathParam($path, $startWithSlash = true)
{
	if (stristr($path, 'Program Files') && stristr($path, '/Git/')) {
		$pos = stripos($path, '/Git/');
		$path = substr($path, $pos + 4);
	}
	if ($startWithSlash && $path[0] != '/') {
		$path = '/'.$path;
	}
	return $path;
}

function createMappedClass($className)
{
$template = '<?php

namespace Website\Model;

class %s extends \Pimcore\Model\Object\%s 
{

	/*								Helpers										*/

	/*								Mapped										*/

	/* 								For elastic search							*/

}
';

	$modelPath = sprintf('%s/models/%s.php', PIMCORE_WEBSITE_PATH, $className);
	try {
		if (!file_exists($modelPath)) {
			\Pimcore\File::put($modelPath, vsprintf($template, [
				$className,
				$className
			]));
		}
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: file [%s] not writable\n", $modelPath));
		exit;
	}

	return true;
}

function createMappedClassList($className)
{
$template = '<?php

namespace Website\Model\%s;

class Listing extends \Pimcore\Model\Object\%s\Listing
{

}
';

	$modelPath = sprintf('%s/models/%s/Listing.php', PIMCORE_WEBSITE_PATH, $className);
	try {
		if (!file_exists($modelPath)) {
			\Pimcore\File::put($modelPath, vsprintf($template, [
				$className,
				$className
			]));
		}
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: file [%s] not writable\n", $modelPath));
		exit;
	}

	return true;
}

function deleteMappedClass($className)
{
	$modelPath = PIMCORE_WEBSITE_PATH.'/models/'.$className.'.php';
	try {
		if (file_exists($modelPath)) {
			unlink($modelPath);
		}
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not delete file [%s]\n", $modelPath));
		exit;
	}

	return true;
}

function deleteMappedClassList($className)
{
	$modelsDir = PIMCORE_WEBSITE_PATH.'/models';
	$modelPath = sprintf('%s/%s/Listing.php', $modelsDir, $className);
	try {
		if (file_exists($modelPath)) {
			unlink($modelPath);
		}
		if (file_exists($modelsDir.'/'.$className)) {
			rmdir($modelsDir.'/'.$className);
		}
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not delete file [%s] or directory [%s]\n", $modelPath, $modelsDir.'/'.$className));
		exit;
	}

	return true;
}

function createAction($name, $controller, $form = null, $view = true, $forceExistenceError = true)
{
	if (!$controller) $controller = 'Cms';
	$tokens = loadControllerTokens($controller);
	$actionExists = checkIfTokenExists($tokens, $name.'Action', $forceExistenceError);

	if (!$actionExists) {
		$sectionIndex = findControllerSectionTokenIndex($tokens, DOCUMENT_ACTIONS);
		$actionContent = createControllerAction($name, $form);
		array_splice($tokens, $sectionIndex, 0, $actionContent);
	}

	$scriptPath = sprintf('%s/views/scripts/%s/%s.php', PIMCORE_WEBSITE_PATH, lcfirst($controller), normalizeScriptFileName($name));
	if ($view && !file_exists($scriptPath)) {
		$scriptContents = ($view === true) ? "\n\n<?\$this->layout()->setLayout('standard');?>\n" : file_get_contents(PIMCORE_DOCUMENT_ROOT . $view);
		try {
			\Pimcore\File::put($scriptPath, $scriptContents);
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: view script [%s] can not be saved\n"));
			exit;
		}
	}

	if (!$actionExists) return saveControllerTokens($controller, $tokens);
	else return true;
}

function createFormHandler($name, $controller)
{
	if (!$controller) $controller = 'Cms';
	$tokens = loadControllerTokens($controller);
	checkIfTokenExists($tokens, 'handle'.$name.'Form');

	$sectionIndex = findControllerSectionTokenIndex($tokens, FORM_ACTIONS);
	$actionContent = createControllerFormHandler($name);

	array_splice($tokens, $sectionIndex, 0, $actionContent);

	return saveControllerTokens($controller, $tokens);
}

function loadControllerTokens($controllerName)
{
	$path = sprintf('%s/controllers/%sController.php', PIMCORE_WEBSITE_PATH, $controllerName);

	$fileContents = file_get_contents($path);

	if (!$fileContents) {
		fputs(STDOUT, sprintf("Error: controller [%s] can not be loaded\n", $path));
		exit;
	}

	return token_get_all($fileContents);
}

function saveControllerTokens($controllerName, $tokens)
{
	$path = sprintf('%s/controllers/%sController.php', PIMCORE_WEBSITE_PATH, $controllerName);

	$fileContents = '';
	foreach($tokens as $token) {
		if (is_string($token)) {
			$fileContents .= $token;
		} else {
			list(, $text) = $token;
			$fileContents .= $text;
		}
	}

	try {
		\Pimcore\File::put($path, $fileContents);
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: controller [%s] could not be saved\n", $path));
		exit;
	}

	return true;
}

function checkIfTokenExists($tokens, $tokenPart, $forceExit = true)
{
	foreach($tokens as $token) {
		$tokenString = null;
		if (is_string($token)) {
			$tokenString .= $token;
		} else {
			list(, $tokenString) = $token;
		}
		if (stristr($tokenString, $tokenPart)) {
			if ($forceExit) {
				fputs(STDOUT, sprintf("Error: token [%s] already exists\n", $tokenPart));
				exit;
			}
			return true;
		}
	}

	return false;
}

function findControllerSectionTokenIndex($tokens, $section)
{
	$sectionFound = false;
	foreach($tokens as $key => $token) {
		$text = null;
		if (is_string($token)) $text = $token;
		else list(, $text) = $token;
		$text = trim($text);

		if (strstr($text, $section) !== false) {
			$sectionFound = true;
		} elseif ($sectionFound && !empty($text)) {
			return $key;
		}
	}

	fputs(STDOUT, sprintf("Error: section [%s] not found\n", $section));
	exit;
}

function createControllerAction($name, $form = null)
{
$template = 'public function %sAction()
	{
		$this->enableLayout();%s
	}

	';
$formHandlerTemplate = '
		$form = new \Website\Form\%sForm(); 

		if ($this->getRequest()->isPost()) {
			if ($this->handle%sForm($form)) {
				$this->gotoUrl($this->view->docUrl(\'/\'.$this->defaultLanguage));
			}
		}

		$this->view->%sForm = $form;';

	return vsprintf($template, [
		$name,
		($form) ? vsprintf($formHandlerTemplate, [$form, $form, lcfirst($form)]) : ''
	]);

}

function createControllerFormHandler($name)
{
$template = 'private function handle%sForm(\Zend_Form &$form)
	{
		if ($form->isValid($this->getRequest()->getPost())) {
			$data = $form->getValues();

			return true;
		} else {
			$this->addErrorMsg($this->translate(\'msg_form_invalid\'));
		}
		return false;
	}

	';

	return vsprintf($template, [$name]);

}

function createDocumentPage($name = null, $controller = null, $action = null, $parent = null, $view = null, $master = null)
{
	# try to load document title from <h1> if not passed as param
	if (!$name && (!$view)) {
		fputs(STDOUT, sprintf("Error: name param is required when no view script is passed\n"));
		exit;
	}
	if (!$name) {
		$scriptPath = PIMCORE_DOCUMENT_ROOT . $view;
		if (!file_exists($scriptPath)) {
			fputs(STDOUT, sprintf("Error: view script [%s] not found\n", $scriptPath));
			exit;
		}
		$matches = [];
		preg_match('/<h1.*>(.*)<\/h1>.*/iU', file_get_contents($scriptPath), $matches);
		if (empty($matches)) {
			fputs(STDOUT, sprintf("Error: <h1> element not found in view script [%s]\n", $scriptPath));
			exit;
		}
		$name = $matches[1];
	}

	# check parent or load the first under "home" 
	$parentId = null;
	if (!$parent) {
		$parent = current(\Pimcore\Model\Document::getById(1)->getChilds());
		if ($parent) $parentId = $parent->getId();
	} else {
		$tmp = null;
		if (intval($parent)) $tmp = \Pimcore\Model\Document::getById(intval($parent));
		else {
			$parent = normalizePathParam($parent, false);
			$tmp = \Pimcore\Model\Document::getByPath($parent);
		}
		if ($tmp) $parentId = $tmp->getId();
	}
	if (!$parentId) {
		fputs(STDOUT, sprintf("Error: can not load parent document [%s]\n", $parent));
		exit;
	}

	# check content document master
	$masterId = null;
	if ($master) {
		$tmp = null;
		if (intval($master)) $tmp = \Pimcore\Model\Document\Page::getById(intval($master));
		else {
			$master = normalizePathParam($master, false);
			$tmp = \Pimcore\Model\Document\Page::getByPath($master);
		}
		if ($tmp) $masterId = $tmp->getId();
		if (!$tmp) {
			fputs(STDOUT, sprintf("Error: could not load master document [%s]\n", $master));
			exit;
		}
	}

	# check controller
	if (!$controller) $controller = 'Cms';
	$controller = ucfirst($controller);
	$controllerPath = PIMCORE_WEBSITE_PATH.'/controllers/'.$controller.'Controller.php';
	if (!file_exists($controllerPath)) {
		fputs(STDOUT, sprintf("Error: controller [%s] does not exists\n", $controllerPath));
		exit;
	}

	# check or try to create action
	if (!$action) $action = 'templateSimple';
	if (!$view) $view = true;
	createAction($action, $controller, null, $view, false);

	# create document
	$document = new \Pimcore\Model\Document\Page();
	$document->setIndex(0);
	$document->setPublished(true);
	$document->setController(normalizeScriptFileName($controller));
	$document->setAction(normalizeScriptFileName($action));
	$document->setParentId($parentId);
	$document->setTitle($name);
	$document->setKey(\Website\Tool\Utils::webalize($name));
	$document->setProperty('navigation_name', 'text', $name, false, false);
	$document->setProperty('navigation_title', 'text', $name, false, false);
	if ($masterId) $document->setContentMasterDocumentId($masterId);
	try {
		$document->save();
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not save document [%s]\n", $e->getMessage()));
		exit;
	}

	return $document;
}

function createDocumentEmail($name, $parent = null)
{

	# check parent or load the first under "/_default_language_/notifikace"
	$parentId = null;
	if (!$parent) {
		$notificationsFolderPath = sprintf('/%s/notifikace', \Website\Tool\Utils::getDefaultLanguage());
		$notificationsFolder = \Pimcore\Model\Document\Folder::getByPath($notificationsFolderPath);
		if ($notificationsFolder) $parentId = $notificationsFolder->getId();
	} else {
		$tmp = null;
		if (intval($parent)) $tmp = \Pimcore\Model\Document::getById(intval($parent));
		else {
			$parent = normalizePathParam($parent, false);
			$tmp = \Pimcore\Model\Document::getByPath($parent);
		}
		if ($tmp) $parentId = $tmp->getId();
	}
	if (!$parentId) {
		fputs(STDOUT, sprintf("Error: can not load document parent [%s]\n", $parent));
		exit;
	}

	$webTitle = \Pimcore\Model\Document\Page::getById(1)->getProperty('titlePostfix');
	$email = new Pimcore\Model\Document\Email();
	$email->setParentId($parentId);
	$email->setKey(\Website\Tool\Utils::webalize($name));
	$email->setPublished(true);
	$email->setController('mail');
	$email->setAction('simple-mail');
	$email->setFrom('noreply@vm3.portadesign.cz');
	$email->setSubject(sprintf('%s - %s', $webTitle, $name));
	$email->setIndex(0);
	try {
		$email->save();
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not create email [%s], reason [%s]\n", $name, $e->getMessage()));
		exit;
	}
	$rte = Pimcore\Model\Document\Tag::factory('wysiwyg', 'email_body', $email->getId());

	$rte->text = sprintf('<p>Dobrý den,</p><p>Text macro example: %%Text(textVariableName);</p><p>Object macro example: %%Object(objectVariableName, {"method":"getMethodName"});</p><p>%s</p>', $webTitle);
	$email->setElement('email_body', $rte);
	$email->save();

	return $email;
}

function createArea($name, $icon)
{
	# create folder
	$areaFolderPath = sprintf('%s/views/areas/%s', PIMCORE_WEBSITE_PATH, $name);
	if (file_exists($areaFolderPath)) {
		fputs(STDOUT, sprintf("Error: area [%s] already exists\n", $name));
		exit;
	}
	\Pimcore\File::mkdir($areaFolderPath);

	# create area config
	$config = loadXmlConfig(null);
	$config->id = $name;
	$config->name = 'area_'.$name;
	$config->description = 'area_'.$name.'_description';
	$config->icon = ($icon) ? $icon : '/pimcore/static/img/icon/html.png';
	$config->version = '1.0';
	writeXmlConfig($config, $areaFolderPath . "/area.xml");

	# enable area in extensions.xml
	$extensionsPath = PIMCORE_WEBSITE_VAR.'/config/extensions.xml';
	$exConfig = loadXmlConfig($extensionsPath);
	if (!isset($exConfig->brick->$name)) $exConfig->brick->$name = 1;
	writeXmlConfig($exConfig, $extensionsPath);

	# create action.php and view.php files
$actionTemplate = '<?php

namespace Pimcore\Model\Document\Tag\Area;

class %s extends AbstractArea
{
	public function getBrickHtmlTagOpen($brick){
		return \'\';
	}

	public function getBrickHtmlTagClose($brick){
		return \'\';
	}
}
';
	try {
		\Pimcore\File::put($areaFolderPath.'/action.php', sprintf($actionTemplate, $name));
		\Pimcore\File::put($areaFolderPath.'/view.php', "\n");
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not create action [%s] files\n", $name));
		exit;
	}

	return true;
}

function deleteArea($name)
{
	$areaFolderPath = sprintf('%s/views/areas/%s', PIMCORE_WEBSITE_PATH, $name);

	# remove files
	try {
		foreach (scandir($areaFolderPath) as $fileOrFolder) if (!in_array($fileOrFolder, ['.', '..'])) {
			unlink($areaFolderPath.'/'.$fileOrFolder);
		}
		rmdir($areaFolderPath);
	} catch (Exception $e) {
		fputs(STDOUT, sprintf("Error: could not remove action [%s] files\n", $name));
		exit;
	}

	# disable area in extensions.xml
	$extensionsPath = PIMCORE_WEBSITE_VAR.'/config/extensions.xml';
	$exConfig = loadXmlConfig($extensionsPath);
	if (isset($exConfig->brick->$name)) unset($exConfig->brick->$name);
	writeXmlConfig($exConfig, $extensionsPath);

	return true;
}

function createThumbnailPipeline($name, $method, $width, $height)
{
	$pipelinesFolderPath = PIMCORE_WEBSITE_VAR.'/config/imagepipelines';
	if (!file_exists($pipelinesFolderPath)) \Pimcore\File::mkdir($pipelinesFolderPath);

	$pipelinePath = $pipelinesFolderPath.'/'.$name.'.xml';

	if (file_exists($pipelinePath)) {
		fputs(STDOUT, sprintf("Error: file [%s] already exists\n", $pipelinePath));
		exit;
	}

	if (!$method) $method = 'cover';
	$allowedMethods = [
		'cover' => ['width', 'height'],
		'frame' => ['width', 'height'],
		'scaleByWidth' => ['width'],
		'scaleByHeight' => ['height']
	];
	if (!in_array($method, array_keys($allowedMethods))) {
		fputs(STDOUT, sprintf("Error: thumbnail method [%s] not supported, allowed methods: [%s]\n", $method, implode(', ', array_keys($allowedMethods))));
		exit;
	}

	$config = loadXmlConfig();
	$config->name = 'NewsPreviewImage';
	$config->description = '';
	$config->format = 'SOURCE';
	$config->quality = 100;
	$config->highResolution = '';
	$config->filenameSuffix = '';
	$config->items = [];
	$config->items->item = [];
	$config->items->item->method = $method;
	$config->items->item->arguments = [];
	foreach($allowedMethods[$method] as $param) {
		if (!${$param}) {
			fputs(STDOUT, sprintf("Error: param [%s] is required for [%s] thumbnail method\n", $param, $method));
			exit;
		}
		$config->items->item->arguments->$param = ${$param};
	}
	if ($method == 'cover') {
		$config->items->item->arguments->positioning = 'center';
		$config->items->item->arguments->doNotScaleUp = 1;
	}
	writeXmlConfig($config, $pipelinePath);

	return true;
}

function deleteThumbnailPipeline($name)
{
	$pipelinePath = PIMCORE_WEBSITE_VAR.'/config/imagepipelines/'.$name.'.xml';
	if (file_exists($pipelinePath)) {
		unlink($pipelinePath);
	}

	return true;
}
