<?php

/**
 * Base controller set up for multi language web sites.
 * Includes basic stuff we use on our sites (breadcrumbs, manual output caching, some helper methods - could be moved to action helpers...).
 */
namespace Website\Controller;

class BaseController extends \Pimcore\Controller\Action\Frontend
{
	protected $language;
	protected $defaultLanguage;

	/** @var \Pimcore\Translate\Website */
	protected $translator = null;
	protected $currentUrl = null;
	/**
	 * container for breadcrumbs extension (for dynamic pages).
	 *
	 * @var array
	 */
	protected $addToBreadcrumbs = [];
	protected $disableBreadcrumbs = false;
	/**
	 * container for related pages in other system registered langeages.
	 *
	 * @var array
	 */
	protected $languageSwitch = [];
	protected $disableLanguageSwitch = false;
	protected $cacheKeySuffix = '';

	/** @var \Website\Controller\Action\Helper\AjaxResponseBuilder */
	protected $ajaxResponseBuilder = null;

	/**
	 * initialization and global stuff goes here.
	 *
	 * inits session
	 * sets up languages
	 * sets up cache prefixes
	 */
	public function init()
	{
		parent::init();

		//set doctype
		$this->view->doctype('HTML5');
		//add view helpers
		$this->view->addHelperPath(PIMCORE_WEBSITE_PATH.'/views/helpers', '\\Website\\View\\Helper');
		//add action helpers
		\Zend_Controller_Action_HelperBroker::addPath(PIMCORE_WEBSITE_PATH.'/controllers/Action/Helper', '\\Website\\Controller\\Action\\Helper');
		//to avoid cache misses in some cases
		\Pimcore\Cache::setForceImmediateWrite(true);
		//language setup
		if (\Zend_Registry::isRegistered('Zend_Locale')) {
			$locale = \Zend_Registry::get('Zend_Locale');
		} else {
			$language = $this->document->getProperty('language');
			//just in case the document property is not set, should not happen though
			if (!$language) {
				$language = \Website\Tool\Utils::getDefaultLanguage();
			}

			$locale = new \Zend_Locale($language);
			\Zend_Registry::set('Zend_Locale', $locale);
		}
		$this->view->language = $locale->getLanguage();
		$this->language = $locale->getLanguage();
		$this->defaultLanguage = \Website\Tool\Utils::getDefaultLanguage();
		$this->view->defaultLanguage = $this->defaultLanguage;
		//current url
		$this->currentUrl = strtok($this->getRequest()->getRequestUri(), '?');
		$this->view->currentUrl = $this->currentUrl;
		//cache keys for scripts
		$cacheKeySuffix = $this->language.'_'.$this->currentUrl;
		if (\Pimcore\Model\Site::isSiteRequest()) {
			$cacheKeySuffix = \Pimcore\Model\Site::getCurrentSite()->id.'_'.$cacheKeySuffix;
		}
		$this->cacheKeySuffix = \Website\Tool\Utils::toCacheKey($cacheKeySuffix);
		$this->view->cacheKey = $this->cacheKeySuffix;
		$this->view->breadcrumbsCacheKey = '_breadcrumbs_'.$this->cacheKeySuffix;
		$this->view->metaCacheKey = '_meta_'.$this->cacheKeySuffix;
		$this->view->languageSwitchCacheKey = '_language_switch_'.$this->cacheKeySuffix;
		//ajax response builder
		$this->ajaxResponseBuilder = $this->_helper->ajaxResponseBuilder();
		//switch on breadcrubs and switcher cache based on url
		$this->view->doNotCacheLayout = false;
		//set some document editables defaults
		$this->view->width = 0;
	}

	/**
	 * global stuff goes here.
	 *
	 * links the translator
	 * instantiates global forms
	 */
	public function preDispatch()
	{
		parent::preDispatch();

		$this->translator = \Zend_Registry::get('Zend_Translate');
		//global forms
		$this->newsletterForm = new \Website\Form\NewsletterForm();
		$this->newsletterForm->setAttrib('allow-multiple-submissions', true);
		$this->view->newsletterForm = $this->newsletterForm;
	}

	/**
	 * layout-only rendering stuff goes here.
	 *
	 * extends breadcrumbs if the $this->addToBreadcrumbs array is not empty
	 * initializes the language swtich
	 * sets dynamic html meta data
	 * creates dynamic robots.txt
	 */
	public function postDispatch()
	{
		parent::postDispatch();

		//execute only if layout is enabled
		if (\Zend_Layout::getMvcInstance() && \Zend_Layout::getMvcInstance()->isEnabled()) {
			if (!$this->disableBreadcrumbs && ($this->view->doNotCacheLayout || !$this->isOutputCached($this->view->breadcrumbsCacheKey))) {
				//init navigation container for breadcrumbs
				$this->view->breadcrumbsNavigation = null;
				if ($this->document instanceof \Pimcore\Model\Document\Page) {
					/* @var $breadcrumbs \Zend_Navigation */
					$breadcrumbs = $this->view->pimcoreNavigation($this->document);
					$activePages = $breadcrumbs->findAllBy('active', true);
					if (!empty($activePages)) {
						$this->view->breadcrumbsNavigation = $breadcrumbs;
					}
				}

				//extend navigation container for breadcrumbs
				if ($this->view->breadcrumbsNavigation && !empty($this->addToBreadcrumbs)) {
					/* @var $container \Zend_Navigation_Container */
					$container = $this->view->breadcrumbsNavigation;
					$tmp = $container->findAllBy('active', true);
					end($tmp);
					/* @var $deepestActive \Pimcore\Navigation\Page\Uri */
					$deepestActive = current($tmp);
					foreach ($this->addToBreadcrumbs as $key => $page) {
						${'page_'.$key} = \Zend_Navigation_Page_Uri::factory([
							'label' => $page['label'],
							'title' => $page['label'],
							'uri' => $page['url'],
							'active' => 1,
						]);
						$deepestActive->addPage(${'page_'.$key});
						$deepestActive = ${'page_'.$key};
					}
				}
			}

			/* language switch */
			if (!$this->disableLanguageSwitch && ($this->view->doNotCacheLayout || !$this->isOutputCached($this->view->languageSwitchCacheKey))) {
				if (empty($this->languageSwitch) && $this->document instanceof \Pimcore\Model\Document\Page) {
					$this->languageSwitch = \Website\Tool\Utils::getLanguageSwitch($this->language, $this->document->getRealFullPath());
				}
				$this->view->languageSwitch = $this->languageSwitch;
			}

			/* default meta tags using view helpers */
			if ($this->view->doNotCacheLayout || !$this->isOutputCached($this->view->metaCacheKey)) {
				$this->view->headTitle()->setSeparator($this->document->getProperty('titleSeparator'));
				if (!(string) $this->view->headTitle()->getContainer()) {
					$this->view->headTitle($this->document->getTitle());
				}
				$this->view->headTitle($this->document->getProperty('titlePostfix'));
				$this->view->headMeta()->setCharset('utf-8');
				$this->view->headMeta()->setName('author', 'portadesign.cz');
				$metaMap = [];
				foreach ($this->view->headMeta()->getContainer() as $item) {
					$metaMap[$item->type.$item->{$item->type}] = true;
				}
				if (!array_key_exists('namedescription', $metaMap) && $this->document->getDescription()) {
					$this->view->headMeta()->setName('description', $this->document->getDescription());
				}
				$this->view->headMeta()->setName('viewport', 'width=device-width,initial-scale=1');
			}
		}
	}

	protected function setCanonicalUrl($url = null)
	{
		if (!$url) {
			$url = $this->canonicalUrl = preg_replace(
				'/\?.*/',
				'',
				$this->view->serverUrl().$this->currentUrl
			);
		}
		if (substr($url, 0, 4) != 'http') {
			$url = $this->view->serverUrl().$url;
		}

		$this->view->headLink([
			'rel' => 'canonical',
			'href' => $url,
		]);
	}

	/**
	 * adds a success flash message.
	 *
	 * @param string $msg the message
	 */
	public function addSuccessMsg($msg)
	{
		$this->_helper->FlashMessenger->setNamespace('success')->addMessage($msg);
	}

	/**
	 * adds an error flash message.
	 *
	 * @param string $msg the message
	 */
	public function addErrorMsg($msg)
	{
		$this->_helper->FlashMessenger->setNamespace('error')->addMessage($msg);
	}

	/**
	 * adds a notice flash message.
	 *
	 * @param string $msg the message
	 */
	public function addInfoMsg($msg)
	{
		$this->_helper->FlashMessenger->setNamespace('info')->addMessage($msg);
	}

	/**
	 * redirects to given URL.
	 *
	 * @param string $url the URL
	 */
	public function gotoUrl($url, $code = null)
	{
		if ($this->getRequest()->getParam('elastic_document_indexing')) {
			return;
		}

		if ($code) {
			$this->_helper->redirector->setCode($code);
		}
		$this->_helper->redirector->gotoUrl($url);
	}

	/**
	 * redirects to given static route.
	 *
	 * @param string $route  name of the static route
	 * @param array  $params the params for placeholders
	 */
	public function gotoRoute($route, $params = [], $code = null)
	{
		$this->gotoUrl(\Website\Tool\Utils::url($route, $params), $code);
	}

	/**
	 * @param string $keyword the translation key
	 *
	 * @return string translated message
	 */
	public function translate($keyword)
	{
		return $this->translator->translate($keyword);
	}

	/**
	 * because document save triggers elasticsearch document indexing,
	 * which would be handled as ajax request.
	 *
	 * @return bool
	 */
	public function isAjax()
	{
		return $this->getRequest()->isXmlHttpRequest() &&
			!\Website\Tool\ElasticSearch::isUpdateRequest() &&
			!$this->getRequest()->getParam('elastic_document_indexing')
		;
	}

	/**
	 * authenticated action protector.
	 */
	protected function checkAuth()
	{
		if ($this->isEditOrAdminMode()) {
			//create a dummy identity if needed
			if (!\Zend_Auth::getInstance()->hasIdentity()) {
				\Zend_Auth::getInstance()->clearIdentity();
				$identity = new \stdClass();
				$identity->email = 'john.doe@example.com';
				\Zend_Auth::getInstance()->getStorage()->write($identity);
				$this->currentUser = $this->getCurrentUser();
			}
		} else {
			if (!$this->currentUser = $this->getCurrentUser()) {
				\Zend_Auth::getInstance()->clearIdentity();
				$loginUrl = $this->view->docUrl('/prihlaseni');
				if (!$this->getRequest()->isXmlHttpRequest) {
					$loginUrl .= '?backlink='.$this->getRequest()->getRequestUri();
				}
				$this->addErrorMsg($this->translate('msg_authentication_required'));
				if ($this->getRequest()->isXmlHttpRequest()) {
					$data = ['status' => 'success', 'snippets' => ['redirect' => ['type' => 'redirect', 'body' => $loginUrl]]];
					$this->_helper->json($data);
					exit;
				}
				$this->gotoUrl($loginUrl);
			}
		}
	}

	/**
	 * tries to set the given url into the referer cookie for previous page purposes.
	 *
	 * @param string $relativeUrl the relative URL
	 * @param bool   $force       force cookie overwrite
	 * @param bool   $seconds     lifetime
	 */
	protected function setPreviousUrl($relativeUrl, $force = false, $seconds = 600)
	{
		if ((!$this->getRequest()->getCookie('referer', false) || $force) && $relativeUrl) {
			setcookie('referer', $relativeUrl, time() + $seconds, '/');
		}
	}

	/**
	 * tries to get the previous URL from referer cookie.
	 *
	 * @param string $defaultUrl default URL in case the cookie is not set
	 *
	 * @return string relative URL
	 */
	protected function getPreviousUrl($defaultUrl = null)
	{
		$referer = $this->getRequest()->getCookie('referer', false);
		$httpReferer = $this->getHttpReferer();

		if ($referer) {
			setcookie('referer', '', time() - 111, '/');
			//return $this->view->docUrl($referer);
			return $referer;
		} elseif ($defaultUrl !== null) {
			return $defaultUrl;
		} elseif ($httpReferer) {
			return $httpReferer;
		} else {
			return $this->view->docUrl('/');
		}
	}

	/**
	 * @return misc referer URL from HTTP_REFERER server option if it is set and from the site domain or false
	 */
	protected function getHttpReferer()
	{
		$parts1 = parse_url($this->view->serverUrl());
		$parts2 = parse_url($this->getRequest()->getServer('HTTP_REFERER', ''));

		if (!isset($parts1['scheme']) || !isset($parts1['host']) || !isset($parts2['scheme']) || !isset($parts2['host'])) {
			return false;
		}

		if ($parts1['scheme'] == $parts2['scheme'] && $parts1['host'] == $parts2['host'] && isset($parts2['path'])) {
			$query = ($parts2['query']) ? '?'.$parts2['query'] : '';

			return ($parts2['path'] != '' && $parts2['path'] != '/') ? $parts2['path'].$query : false;
		}

		return false;
	}

	/**
	 * checks if there is a cache hit for given key.
	 *
	 * @param string $cacheKey
	 *
	 * @return bool
	 */
	protected function isOutputCached($cacheKey)
	{
		$viewCacheKeyPrefix = 'pimcore_viewcache_';

		if (\Pimcore\Tool::isFrontentRequestByAdmin() ||
				\Pimcore\Cache::load($viewCacheKeyPrefix.$cacheKey) === false ||
				$this->getRequest()->getParam('elastic_document_indexing')) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * @return \Pimcore\Model\Object\User|null
	 */
	protected function getCurrentUser($email = false)
	{
		if (\Zend_Auth::getInstance()->hasIdentity()) {
			$email = \Zend_Auth::getInstance()->getIdentity()->email;
		} elseif (!$email) {
			return;
		}

		$list = new \Pimcore\Model\Object\User\Listing();
		$list->setUnpublished(true);
		$list->setCondition(\Pimcore\Db::get()->quoteInto('email = ?', $email));
		$list->setLimit(1);
		$user = $list->current();

		return $user;
	}

	protected function isEditOrAdminMode()
	{
		if ($this->editmode) {
			return true;
		}
		if (\Pimcore\Tool::isFrontentRequestByAdmin()) {
			$admin = \Pimcore\Tool\Authentication::authenticateSession();
			if ($admin instanceof User) {
				return true;
			}
		}

		return false;
	}
}
