<?php

namespace Website\Model;

trait ElasticTrait
{
	private $shortClassName = null;
	private $elasticSearchmapping = null;

	protected function getShortClassName()
	{
		if (!$this->shortClassName) {
			$reflection = new \ReflectionClass($this);
			$this->shortClassName = $reflection->getShortName();
		}

		return $this->shortClassName;
	}

	private function getObjectMapping()
	{
		if (!$this->elasticSearchmapping) {
			$this->elasticSearchmapping = \Website\Tool\ElasticSearch::getObjectMapping(
				$this->getShortClassName()
			);
		}

		return $this->elasticSearchmapping;
	}

	private function getElasticType($lang, $forFulltext = false)
	{
		if ($forFulltext) {
			if ($this instanceof Document\Page) {
				return \Website\Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY.'_'.$lang;
			} else {
				return \Website\Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY.'_'.$lang;
			}
		} else {
			return strtolower($this->getShortClassName()).'_'.$lang;
		}
	}

	public function getElasticBaseData()
	{
		return [
			'id' => (int) parent::getId(),
			'parentId' => (int) parent::getParentId(),
			'published' => (bool) parent::getPublished(),
			'key' => parent::getKey(),
			'type' => $this->getShortClassName(),
		];
	}

	public function elasticSearchUpdate()
	{
		if (parent::getProperty('elastic_search_exclude')) {
			return;
		}

		$mainIndexName = \Website\Tool\ElasticSearch::getMainIndexName();
		$elasticClient = \Website\Tool\ElasticSearch::getClient();
		$mapping = $this->getObjectMapping();

		if (parent::getId()) {
			foreach ($mapping['meta']['languages'] as $lang) {
				$data = $this->elasticSearchLoad($lang);
				if (isset($data['elasticGroup'])) {
					$this->elasticSearchDelete($lang, current($data['elasticGroup'])['id']);
					foreach($data['elasticGroup'] as $key => $groupData) {
						$elasticClient->index([
							'index' => $mainIndexName,
							'type' => $this->getElasticType($lang),
							'id' => $groupData['id'] . '-' . $key,
							'body' => $groupData,
						]);
					}
				} else {
					$elasticClient->index([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'id' => $data['id'],
						'body' => $data,
					]);
				}
				unset($data);

				if (!\Website\Tool\ElasticSearch::isUpdateRequest()) {
					$elasticClient->indices()->refresh(['index' => $mainIndexName]);
				}
			}
		}
	}

	public function elasticSearchDelete($language = null, $groupId = null)
	{
		$mainIndexName = \Website\Tool\ElasticSearch::getMainIndexName();
		$elasticClient = \Website\Tool\ElasticSearch::getClient();
		$mapping = $this->getObjectMapping();

		foreach ($mapping['meta']['languages'] as $lang) {
			if ($language && $language != $lang) {
				continue;
			}
			try {
				if ($groupId) {
					$elasticClient->deleteByQuery([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'body' => ['query' =>  ['term' => ['id' => $groupId]]],
					]);
				} else {
					$elasticClient->delete([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'id' => parent::getId(),
					]);
				}
			} catch (\Exception $e) {
				\Pimcore\Log\Simple::log('elasticsearch',
					'Could not delete entry for object with ID: '
					. parent::getId() . ' and type: ' . $this->getElasticType($lang)
					. '. Exception: ' . $e->getMessage() . "\n" . $e->getTraceAsString()
				);
			}
		}
		if (!\Website\Tool\ElasticSearch::isUpdateRequest()) {
			$elasticClient->indices()->refresh(['index' => $mainIndexName]);
		}
	}

	public function elasticSearchUpdateFulltext()
	{
		if (parent::getProperty('elastic_search_exclude')) {
			return;
		}

		$indexName = \Website\Tool\ElasticSearch::getMainIndexName();
		$elasticClient = \Website\Tool\ElasticSearch::getClient();

		$languages = null;
		if ($this instanceof Document\Page) {
			$language = $this->getProperty('language');
			if (!$language) {
				$language = \Website\Tool\Utils::getDefaultLanguage();
			}
			$languages[] = $language;
		} else {
			$mapping = $this->getObjectMapping();
			$languages = $mapping['meta']['languages'];
		}

		foreach ($languages as $lang) {
			$data = $this->elasticSearchLoadFulltext($lang);
			$elasticClient->index([
				'index' => $indexName,
				'type' => $this->getElasticType($lang, true),
				'id' => $data['id'],
				'body' => $data,
			]);
			unset($data);

			if (!\Website\Tool\ElasticSearch::isUpdateRequest()) {
				$elasticClient->indices()->refresh(['index' => $indexName]);
			}
		}
	}

	public function elasticSearchDeleteFulltext()
	{
		$mainIndexName = \Website\Tool\ElasticSearch::getMainIndexName();
		$elasticClient = \Website\Tool\ElasticSearch::getClient();

		$languages = null;
		if ($this instanceof Document\Page) {
			$language = $this->getProperty('language');
			if (!$language) {
				$language = \Website\Tool\Utils::getDefaultLanguage();
			}
			$languages[] = $language;
		} else {
			$mapping = $this->getObjectMapping();
			$languages = $mapping['meta']['languages'];
		}

		foreach ($languages as $lang) {
			try {
				$elasticClient->delete([
					'index' => $mainIndexName,
					'type' => $this->getElasticType($lang, true),
					'id' => parent::getId(),
				]);
			} catch (\Exception $e) {
				\Pimcore\Log\Simple::log('elasticsearch',
					'Could not delete entry for document or object with ID: '
					.parent::getId().' and type: '.$this->getElasticType($lang, true)
					.'. Exception: '.$e->getMessage()."\n".$e->getTraceAsString()
				);
			}
		}
		if (!\Website\Tool\ElasticSearch::isUpdateRequest()) {
			$elasticClient->indices()->refresh(['index' => $mainIndexName]);
		}
	}

	/**
	 * decomposes a string into parts for better results of prefix suggestions.
	 */
	public function createElasticSuggestion($string, $payload = null)
	{
		$tmp = \Website\Tool\Utils::webalize($string);

		$suggest = ['input' => explode('-', $tmp), 'output' => $string, 'weight' => 1];
		if ($payload) {
			$suggest['payload'] = $payload;
		}

		return $suggest;
	}

	public function normalizeElasticText($text)
	{
		$text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');
		$text = iconv("UTF-8", "UTF-8//IGNORE", $text);
		$dom = \Sunra\PhpSimple\HtmlDomParser::str_get_html(
			'<div>' . html_entity_decode($text) . '</div>'
		);
		if ($dom) {
			$tagList = [
				'audio', 'area', 'base', 'canvas', 'embed', 'head', 'link', 'map', 'meta',
				'noscript', 'object', 'param', 'script', 'source', 'style', 'svg', 'track', 'video'
			];
			foreach ($tagList as $tag) {
				foreach ($dom->find($tag) as $node) {
					$node->outertext = '';
				}
			}

			$text = $dom->save();
		}

		return preg_replace('/[ \t\r\n]+/', ' ', strip_tags($text));
	}

	public function renderView($template, $language, $params)
	{
		$translatorBackup = null;
		if (\Zend_Registry::isRegistered('Zend_Translate')) {
			$translatorBackup = \Zend_Registry::get('Zend_Translate');
		}
		$translator = new \Pimcore\Translate\Website($language);
		\Zend_Registry::set('Zend_Translate', $translator);

		$view = new \Pimcore\View();
		$view->setScriptPath([
			PIMCORE_WEBSITE_PATH.'/views/scripts',
			PIMCORE_WEBSITE_PATH.'/views/layouts',
		]);
		$view->addHelperPath(PIMCORE_PATH.'/lib/Pimcore/View/Helper', '\\Pimcore\\View\\Helper\\');
		$view->addHelperPath(PIMCORE_WEBSITE_PATH.'/views/helpers', '\\Website\\View\\Helper');
		$view->language = $language;
		foreach ($params as $key => $value) {
			$view->$key = $value;
		}

		$content = $this->normalizeElasticText($view->render($template));
		\Zend_Registry::set('Zend_Translate', $translatorBackup);

		return $content;
	}
}
